"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReloadPlanCommand = exports.PlanDataLoader = void 0;
const data_1 = require("../data");
const logger_1 = require("../logger");
/**
 * Plan 数据加载工具类（纯静态方法）
 */
class PlanDataLoader {
    /**
     * 从文件系统读取完整的 Plan 数据
     * @param context VS Code 扩展上下文
     * @param conversationId 对话ID
     * @returns Plan数据，如果不存在则返回 null
     */
    static async loadPlan(context, conversationId) {
        try {
            // 获取当前扩展的 globalStorage 路径
            const currentGlobalStoragePath = context.globalStorageUri.fsPath;
            // 替换为主扩展 (tencent-cloud.coding-copilot) 的路径
            // 路径格式: .../User/globalStorage/vscode.lexical-markdown-editor
            // 需要改为: .../User/globalStorage/tencent-cloud.coding-copilot
            const mainExtensionPath = currentGlobalStoragePath.replace(/vscode\.lexical-markdown-editor$/, 'tencent-cloud.coding-copilot');
            const planResult = await (0, data_1.getLocalPlanResult)(mainExtensionPath, conversationId);
            return planResult;
        }
        catch (error) {
            console.log('No plan data found for conversation:', conversationId, error);
            return undefined;
        }
    }
    static async loadPlanContent(context, conversationId) {
        try {
            // 获取当前扩展的 globalStorage 路径
            const currentGlobalStoragePath = context.globalStorageUri.fsPath;
            // 替换为主扩展 (tencent-cloud.coding-copilot) 的路径
            // 路径格式: .../User/globalStorage/vscode.lexical-markdown-editor
            // 需要改为: .../User/globalStorage/tencent-cloud.coding-copilot
            const mainExtensionPath = currentGlobalStoragePath.replace(/vscode\.lexical-markdown-editor$/, 'tencent-cloud.coding-copilot');
            const planContent = await (0, data_1.getLocalPlanContent)(mainExtensionPath, conversationId);
            return planContent;
        }
        catch (error) {
            console.log('No plan data found for conversation:', conversationId, error);
            return undefined;
        }
    }
}
exports.PlanDataLoader = PlanDataLoader;
/**
 * 刷新 Plan 预览命令（通知已打开的预览重新加载 plan.json）
 */
class ReloadPlanCommand {
    constructor(_context) {
        this._context = _context;
        this.id = 'lexicalMarkdown.reloadPlan';
    }
    /**
     * 注册 webview panel，以便后续能通知刷新
     */
    static registerWebviewPanel(conversationId, panel) {
        this.webviewPanels.set(conversationId, panel);
        // 当 panel 关闭时，从 Map 中移除
        panel.onDidDispose(() => {
            this.webviewPanels.delete(conversationId);
        });
    }
    /**
     * 执行刷新命令
     */
    async execute(args) {
        const { conversationId, planResult } = args;
        try {
            // 查找对应的 webview panel
            const panel = ReloadPlanCommand.webviewPanels.get(conversationId);
            if (!panel) {
                logger_1.Logger.warn('No active webview found for conversation:', conversationId);
                return;
            }
            // 1. 从文件重新读取 plan 数据
            const planData = {
                ...planResult,
                content: await PlanDataLoader.loadPlanContent(this._context, conversationId)
            };
            if (!planData) {
                logger_1.Logger.warn('No plan data found for conversation:', conversationId);
                return;
            }
            if (!planData.content) {
                logger_1.Logger.warn('No plan content found for conversation:', conversationId);
                return;
            }
            // 2. 计算 diff 数据
            const currentGlobalStoragePath = this._context.globalStorageUri.fsPath;
            const mainExtensionPath = currentGlobalStoragePath.replace(/vscode\.lexical-markdown-editor$/, 'tencent-cloud.coding-copilot');
            const diffResult = await (0, data_1.calculateCurrentDiff)(mainExtensionPath, conversationId);
            // 3. 发送刷新消息到 webview（包含 diff 数据）
            await panel.webview.postMessage({
                type: 'reloadPlan',
                planResult: planData,
                diffStats: diffResult?.stats,
                diffChanges: diffResult?.changes
            });
        }
        catch (error) {
            logger_1.Logger.error('Failed to reload plan data:', error);
        }
    }
}
exports.ReloadPlanCommand = ReloadPlanCommand;
// 保存打开的 webview 引用，key 为 conversationId
ReloadPlanCommand.webviewPanels = new Map();
//# sourceMappingURL=syncPlan.js.map